<?php
declare(strict_types=1);
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/layout.php';
require_motoboy();

$motoboyId = (int)$_SESSION['motoboy_id'];

$st = $pdo->prepare("SELECT is_online, last_lat, last_lng FROM motoboys WHERE id = ?");
$st->execute([$motoboyId]);
$m = $st->fetch();
$isOnline = (int)($m['is_online'] ?? 0);

$radiusKm = 8; // ajuste o raio aqui
$available = [];

if ($isOnline) {
  $mLat = $m['last_lat'] ?? null;
  $mLng = $m['last_lng'] ?? null;

  if ($mLat && $mLng) {
    $sql = "
      SELECT
        d.*,
        c.name AS company_name,
        (6371 * acos(
          cos(radians(:mlat)) * cos(radians(d.pickup_lat)) *
          cos(radians(d.pickup_lng) - radians(:mlng)) +
          sin(radians(:mlat)) * sin(radians(d.pickup_lat))
        )) AS distance_km
      FROM deliveries d
      JOIN companies c ON c.id = d.company_id
      WHERE d.status = 'waiting'
        AND d.pickup_lat IS NOT NULL AND d.pickup_lng IS NOT NULL
      HAVING distance_km <= :radius
      ORDER BY distance_km ASC
      LIMIT 50
    ";
    $st = $pdo->prepare($sql);
    $st->execute([':mlat'=>$mLat, ':mlng'=>$mLng, ':radius'=>$radiusKm]);
    $available = $st->fetchAll();
  } else {
    $st = $pdo->query("
      SELECT d.*, c.name AS company_name
      FROM deliveries d
      JOIN companies c ON c.id = d.company_id
      WHERE d.status = 'waiting'
      ORDER BY d.id ASC
      LIMIT 50
    ");
    $available = $st->fetchAll();
  }
}

$st = $pdo->prepare("
  SELECT d.*, c.name AS company_name
  FROM deliveries d
  JOIN companies c ON c.id = d.company_id
  WHERE d.motoboy_id = ?
  ORDER BY d.id DESC
  LIMIT 50
");
$st->execute([$motoboyId]);
$mine = $st->fetchAll();

render_header('Motoboy - Entregas');
?>
<div class="grid grid-2">
  <div class="card">
    <h1>Disponíveis</h1>
    <p class="small">Requer online. Se sua posição estiver atualizada, ordena por distância (raio <?= (int)$radiusKm ?> km).</p>
    <?php if (!$isOnline): ?>
      <div class="notice">Você está offline. Volte ao dashboard e fique online.</div>
    <?php elseif (!$available): ?>
      <div class="notice">Nenhuma entrega aguardando no momento (ou fora do raio).</div>
    <?php else: ?>
      <table class="table">
        <thead>
          <tr><th>#</th><th>Empresa</th><th>Coleta</th><th>Dist.</th><th>Ação</th></tr>
        </thead>
        <tbody>
        <?php foreach ($available as $d): ?>
          <tr>
            <td><?= (int)$d['id'] ?></td>
            <td><?= h($d['company_name']) ?></td>
            <td><?= h($d['pickup_address']) ?></td>
            <td><?= isset($d['distance_km']) ? number_format((float)$d['distance_km'], 2) . " km" : "-" ?></td>
            <td><button class="btn btn-ok" onclick="accept(<?= (int)$d['id'] ?>)">Aceitar</button></td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
    <?php endif; ?>
  </div>

  <div class="card">
    <h1>Minhas entregas</h1>
    <?php if (!$mine): ?>
      <div class="notice">Você ainda não aceitou entregas.</div>
    <?php else: ?>
      <table class="table">
        <thead>
          <tr><th>#</th><th>Status</th><th>Entrega</th><th>Ações</th></tr>
        </thead>
        <tbody>
        <?php foreach ($mine as $d): ?>
          <tr>
            <td><?= (int)$d['id'] ?></td>
            <td><?= h($d['status']) ?></td>
            <td><?= h($d['dropoff_address']) ?></td>
            <td style="white-space:nowrap;">
              <a class="btn btn-ghost" href="track.php?t=<?= h($d['tracking_token']) ?>" target="_blank">Link cliente</a>
            </td>
          </tr>
          <tr>
            <td colspan="4">
              <div class="row">
                <select onchange="setStatus(<?= (int)$d['id'] ?>, this.value)">
                  <?php
                    $opts = ['accepted','picked_up','delivering','delivered','canceled'];
                    foreach ($opts as $o) {
                      $sel = ($d['status'] === $o) ? 'selected' : '';
                      echo "<option value=\"".h($o)."\" $sel>".h($o)."</option>";
                    }
                  ?>
                </select>
                <button class="btn btn-primary" onclick="startLocation(<?= (int)$d['id'] ?>)">Enviar localização (5s)</button>
              </div>

              <div class="small" id="locmsg-<?= (int)$d['id'] ?>" style="margin-top:6px;"></div>

              <form onsubmit="return uploadProof(event, <?= (int)$d['id'] ?>)" enctype="multipart/form-data" style="margin-top:10px;">
                <div class="row">
                  <input type="file" name="photo" accept="image/*" required>
                  <button class="btn btn-primary" type="submit">Enviar foto</button>
                </div>
                <div class="small" id="proofmsg-<?= (int)$d['id'] ?>"></div>
              </form>
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
    <?php endif; ?>
  </div>
</div>

<script>
async function accept(id){
  const res = await fetch('api/accept_delivery.php', {
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'delivery_id='+encodeURIComponent(id)
  });
  const j = await res.json();
  if(j.ok) location.reload();
  else alert(j.error || 'Erro');
}

async function setStatus(id, status){
  const res = await fetch('api/update_status.php', {
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'delivery_id='+encodeURIComponent(id)+'&status='+encodeURIComponent(status)
  });
  const j = await res.json();
  if(!j.ok) alert(j.error || 'Erro');
}

const timers = {};
function startLocation(deliveryId){
  if(timers[deliveryId]) return;

  const el = document.getElementById('locmsg-'+deliveryId);
  el.textContent = 'Iniciando GPS...';

  timers[deliveryId] = setInterval(async () => {
    if(!navigator.geolocation){
      el.textContent = 'Seu navegador não suporta geolocalização.';
      clearInterval(timers[deliveryId]); timers[deliveryId]=null;
      return;
    }

    navigator.geolocation.getCurrentPosition(async (pos) => {
      const lat = pos.coords.latitude;
      const lng = pos.coords.longitude;

      const res = await fetch('api/update_location.php', {
        method:'POST',
        headers:{'Content-Type':'application/x-www-form-urlencoded'},
        body:'delivery_id='+encodeURIComponent(deliveryId)+'&lat='+encodeURIComponent(lat)+'&lng='+encodeURIComponent(lng)
      });
      const j = await res.json();
      el.textContent = j.ok ? ('Localização enviada: '+lat.toFixed(5)+', '+lng.toFixed(5)) : (j.error || 'Erro');
    }, (err) => {
      el.textContent = 'Erro ao obter localização: '+err.message;
    }, { enableHighAccuracy:true, timeout:8000 });
  }, 5000);
}

async function uploadProof(ev, deliveryId){
  ev.preventDefault();
  const form = ev.target;
  const msg = document.getElementById('proofmsg-'+deliveryId);

  const fd = new FormData(form);
  fd.append('delivery_id', deliveryId);

  msg.textContent = 'Enviando...';
  const res = await fetch('api/upload_proof.php', { method:'POST', body: fd });
  const j = await res.json();
  msg.textContent = j.ok ? ('Foto enviada: '+j.path) : (j.error || 'Erro');
  return false;
}
</script>

<?php render_footer(); ?>
